// Copyright 2026 Danslav Slavenskoj, Lingenic LLC
// License: CC0 1.0 - Public Domain
// https://creativecommons.org/publicdomain/zero/1.0/
// You may use this code for any purpose without attribution.
//
// Spec: https://hsvfile.com
// Repo: https://github.com/LingenicLLC/HSV

// Package hsv implements parsing for HSV (Hierarchical Separated Values),
// a text-based file format and streaming protocol using ASCII control characters.
package hsv

import (
	"strings"
)

// Control characters
const (
	SOH = '\x01' // Start of Header
	STX = '\x02' // Start of Text (data block)
	ETX = '\x03' // End of Text
	EOT = '\x04' // End of Transmission
	SO  = '\x0e' // Shift Out (start nested)
	SI  = '\x0f' // Shift In (end nested)
	DLE = '\x10' // Data Link Escape (binary mode)
	FS  = '\x1c' // File/Record Separator
	GS  = '\x1d' // Group/Array Separator
	RS  = '\x1e' // Record/Property Separator
	US  = '\x1f' // Unit/Key-Value Separator
)

// Value represents an HSV value - string, array, or nested object
type Value interface{}

// Object represents a key-value map
type Object map[string]Value

// Array represents a list of values
type Array []Value

// Document represents a parsed HSV document
type Document struct {
	Header  Object
	Records []Object
}

// Parse parses HSV text into a Document
func Parse(text string) Document {
	// Extract binary sections first
	text, binaries := extractBinarySections(text)
	runes := []rune(text)

	doc := Document{
		Header:  nil,
		Records: []Object{},
	}

	i := 0
	for i < len(runes) {
		// Check for SOH (header start)
		if runes[i] == SOH {
			// Find STX (header end, data start)
			stxPos := indexRune(runes[i+1:], STX)
			if stxPos == -1 {
				i++
				continue
			}
			stxPos += i + 1

			// Parse header
			headerContent := string(runes[i+1 : stxPos])
			doc.Header = parseObject(headerContent, binaries)

			// Find ETX (data end)
			etxPos := indexRune(runes[stxPos+1:], ETX)
			if etxPos == -1 {
				i = stxPos + 1
				continue
			}
			etxPos += stxPos + 1

			// Parse data block
			dataContent := string(runes[stxPos+1 : etxPos])
			for _, record := range splitRespectingNesting(dataContent, FS) {
				obj := parseObject(record, binaries)
				if len(obj) > 0 {
					doc.Records = append(doc.Records, obj)
				}
			}

			i = etxPos + 1
		} else if runes[i] == STX {
			// Data block without header
			etxPos := indexRune(runes[i+1:], ETX)
			if etxPos == -1 {
				i++
				continue
			}
			etxPos += i + 1

			dataContent := string(runes[i+1 : etxPos])
			for _, record := range splitRespectingNesting(dataContent, FS) {
				obj := parseObject(record, binaries)
				if len(obj) > 0 {
					doc.Records = append(doc.Records, obj)
				}
			}

			i = etxPos + 1
		} else {
			i++
		}
	}

	return doc
}

func indexRune(runes []rune, r rune) int {
	for i, c := range runes {
		if c == r {
			return i
		}
	}
	return -1
}

func extractBinarySections(text string) (string, map[string]string) {
	var result strings.Builder
	binaries := make(map[string]string)
	runes := []rune(text)
	i := 0
	placeholderCount := 0

	for i < len(runes) {
		// Check for DLE+STX (binary start)
		if runes[i] == DLE && i+1 < len(runes) && runes[i+1] == STX {
			j := i + 2
			var binaryData strings.Builder

			for j < len(runes) {
				if runes[j] == DLE && j+1 < len(runes) {
					if runes[j+1] == ETX {
						// End of binary section
						placeholder := string(rune(0)) + "BINARY" + string(rune(placeholderCount)) + string(rune(0))
						binaries[placeholder] = unescapeBinary(binaryData.String())
						result.WriteString(placeholder)
						placeholderCount++
						i = j + 2
						break
					} else if runes[j+1] == DLE {
						// Escaped DLE
						binaryData.WriteRune(DLE)
						j += 2
						continue
					}
				}
				binaryData.WriteRune(runes[j])
				j++
			}

			if j >= len(runes) {
				result.WriteRune(runes[i])
				i++
			}
		} else {
			result.WriteRune(runes[i])
			i++
		}
	}

	return result.String(), binaries
}

func unescapeBinary(data string) string {
	var result strings.Builder
	runes := []rune(data)
	i := 0

	for i < len(runes) {
		if runes[i] == DLE && i+1 < len(runes) && runes[i+1] == DLE {
			result.WriteRune(DLE)
			i += 2
		} else {
			result.WriteRune(runes[i])
			i++
		}
	}

	return result.String()
}

func restoreBinaries(value string, binaries map[string]string) string {
	for placeholder, data := range binaries {
		value = strings.ReplaceAll(value, placeholder, data)
	}
	return value
}

func splitRespectingNesting(text string, sep rune) []string {
	var parts []string
	var current strings.Builder
	depth := 0

	for _, c := range text {
		switch c {
		case SO:
			depth++
			current.WriteRune(c)
		case SI:
			depth--
			current.WriteRune(c)
		case sep:
			if depth == 0 {
				parts = append(parts, current.String())
				current.Reset()
			} else {
				current.WriteRune(c)
			}
		default:
			current.WriteRune(c)
		}
	}

	if current.Len() > 0 || len(parts) > 0 {
		parts = append(parts, current.String())
	}

	return parts
}

func parseValue(value string, binaries map[string]string) Value {
	value = restoreBinaries(value, binaries)
	runes := []rune(value)

	// Check for nested structure (SO at start, SI at end)
	if len(runes) >= 2 && runes[0] == SO && runes[len(runes)-1] == SI {
		inner := string(runes[1 : len(runes)-1])
		return parseObject(inner, binaries)
	}

	// Check for array
	if strings.ContainsRune(value, GS) {
		parts := splitRespectingNesting(value, GS)
		arr := make(Array, len(parts))
		for i, p := range parts {
			arr[i] = parseValue(p, binaries)
		}
		return arr
	}

	return value
}

func parseObject(content string, binaries map[string]string) Object {
	obj := make(Object)

	props := splitRespectingNesting(content, RS)
	for _, prop := range props {
		parts := splitRespectingNesting(prop, US)
		if len(parts) >= 2 {
			k := parts[0]
			v := strings.Join(parts[1:], string(US))
			obj[k] = parseValue(v, binaries)
		}
	}

	return obj
}
